<?php
/* * ********************************************************************************************
 * 								Open Real Estate
 * 								----------------
 * 	version				:	V1.36.0
 * 	copyright			:	(c) 2016 Monoray
 * 							http://monoray.net
 * 							http://monoray.ru
 *
 * 	website				:	http://open-real-estate.info/en
 *
 * 	contact us			:	http://open-real-estate.info/en/contact-us
 *
 * 	license:			:	http://open-real-estate.info/en/license
 * 							http://open-real-estate.info/ru/license
 *
 * This file is part of Open Real Estate
 *
 * ********************************************************************************************* */

class Seasonalprices extends ParentModel
{

    public $in_currency;
    public $date_start_formatting;
    public $date_end_formatting;
    public $dateStart;
    public $dateEnd;
    public $min_rental_period_with_type;
    public $priceWithType;

    public static function model($className = __CLASS__)
    {
        return parent::model($className);
    }

    public function tableName()
    {
        return '{{seasonal_prices}}';
    }

    public function currencyFields()
    {
        return array('price');
    }

    public function rules()
    {
        return array(
            array('price, date_start, month_start, date_end, month_end, price_type', 'required'),
            array('date_start_formatting, date_end_formatting', 'date', 'format' => 'd-m', 'allowEmpty' => false, 'message' => tt('Wrong format of field {attribute}', 'seasonalprices')),
            array('name', 'i18nRequired'),
            array('name', 'i18nLength', 'max' => 255),
            array('apartment_id', 'required', 'on' => 'insert'),
            array('price_type', 'numerical', 'min' => 1),
            array('min_rental_period, sorter', 'numerical'),
            array($this->getI18nFieldSafe(), 'safe'),
            array('dateStart, dateEnd, in_currency', 'safe'),
            array('date_end', 'validDate'),
            array('id, date_start, date_end, min_rental_period, apartment_id', 'safe', 'on' => 'search'),
        );
    }

    public function i18nFields()
    {
        return array(
            'name' => 'varchar(255) not null default ""',
        );
    }

    public function behaviors()
    {
        return array(
            'AutoTimestampBehavior' => array(
                'class' => 'zii.behaviors.CTimestampBehavior',
                'createAttribute' => 'date_created',
                'updateAttribute' => 'date_updated',
            ),
        );
    }

    public function attributeLabels()
    {
        return array(
            'id' => 'Id',
            'name' => tt('Name', 'seasonalprices'),
            'price' => tt('Price', 'seasonalprices'),
            'price_type' => tt('Price_type', 'seasonalprices'),
            'date_start' => tt('From', 'seasonalprices'),
            'date_end' => tt('To', 'seasonalprices'),
            'min_rental_period' => tt('Min_rental_period', 'seasonalprices'),
            'dateStart' => tt('From', 'seasonalprices'),
            'dateEnd' => tt('To', 'seasonalprices'),
            'date_start_formatting' => tt('From', 'seasonalprices'),
            'date_end_formatting' => tt('To', 'seasonalprices'),
        );
    }

    /**
     * Find overlap days in seasons
     */
    public function validDate($attribute, $params)
    {
        $year = 2015;
        $seasonStart = DateTime::createFromFormat('d-m-Y', $this->date_start . '-' . $this->month_start . '-' . $year);
        $seasonEnd = DateTime::createFromFormat('d-m-Y', $this->date_end . '-' . $this->month_end . '-' . $year);

        $addWhere = '';
        if (!$this->isNewRecord) {
            $addWhere = ' AND id != ' . $this->id . ' ';
        }

        $priceRows = Yii::app()->db
            ->createCommand("SELECT name_" . Yii::app()->language . " AS name, month_start, date_start, month_end, date_end, price FROM {{seasonal_prices}}
					WHERE price_type = :t AND apartment_id=:id {$addWhere} ORDER BY price ASC")
            ->queryAll(true, array(
                ':t' => $this->price_type,
                ':id' => $this->apartment_id,
            ));

        if (!empty($priceRows)) {
            foreach ($priceRows as $row) {
                $seasonStart2 = DateTime::createFromFormat('d-m-Y', $row['date_start'] . '-' . $row['month_start'] . '-' . $year);
                $seasonEnd2 = DateTime::createFromFormat('d-m-Y', $row['date_end'] . '-' . $row['month_end'] . '-' . $year);

                $daysOverlap = HBooking::datesOverlap($seasonStart2, $seasonEnd2, $seasonStart, $seasonEnd, 0);
                if ($daysOverlap) {
                    $this->addError('date_end', tt('For these days, the season has already exhibited price', 'seasonalprices'));
                    break;
                }
            }
        }
    }

    public function beforeSave()
    {
        if (issetModule('currency') && isset($this->in_currency) && $this->in_currency) {
            $defaultCurrencyCharCode = Currency::getDefaultCurrencyModel()->char_code;
            if ($defaultCurrencyCharCode != $this->in_currency) {
                $this->price = (int)Currency::convert($this->price, $this->in_currency, $defaultCurrencyCharCode);
            }
        }

        if ($this->isNewRecord) {
            $sql = 'SELECT MAX(sorter) FROM ' . $this->tableName() . ' WHERE apartment_id = "' . $this->apartment_id . '"';
            $maxSorter = Yii::app()->db
                ->createCommand($sql)
                ->queryScalar();
            $this->sorter = $maxSorter + 1;
        }

        return parent::beforeSave();
    }

    public function afterFind()
    {
        $this->priceWithType = self::makePriceWithType($this->price, $this->price_type);

        $this->dateStart = self::makeDate($this->date_start, $this->month_start);
        $this->dateEnd = self::makeDate($this->date_end, $this->month_end);

        $this->min_rental_period_with_type = (!$this->min_rental_period) ? '-' : $this->min_rental_period . ' ' . Seasonalprices::rentalPeriodNames($this->price_type);

        parent::afterFind();
    }

    public static function makeDate($date = "", $month = "")
    {
        if ($date && $month) {
            $dateFormat = Yii::app()->locale->getDateFormat('long');
            if (($strpos = mb_strpos($dateFormat, 'y')) !== false) {
                if (Yii::app()->language == 'ru') {
                    $dateFormat = mb_substr($dateFormat, 0, $strpos, 'utf-8'); // так не выводит для lt языка
                }
                $dateFormat = str_replace('y', '', $dateFormat);
            }
            $ret = Yii::app()->dateFormatter->format($dateFormat, CDateTimeParser::parse($date . '-' . $month, 'd-M'));
            if (!$ret) {
                $ret = str_pad($date, 2, '0', STR_PAD_LEFT) . '-' . str_pad($month, 2, '0', STR_PAD_LEFT);
            }
            return $ret;
        } else
            return "";
    }

    public static function makePriceWithType($price, $price_type)
    {
        if ($price && is_numeric($price) && $price_type) {
            $currencyName = (issetModule('currency')) ? Currency::getCurrentCurrencyName() : param('siteCurrency', '$');

            $price = round(issetModule('currency') ? Currency::convertFromDefault($price) : $price, param('round_price', 2));

            if (Yii::app()->theme->name == Themes::THEME_ATLAS_NAME) {
                $tpl = Lang::getTemplateForPrice(Yii::app()->language);
                $data = array(
                    '{CURRENCY}' => $currencyName,
                    '{PRICE}' => Apartment::setPrePretty($price),
                    '{TYPE}' => Seasonalprices::rentalPeriodNames($price_type),
                );

                return strtr($tpl['default'], $data);
            } else {
                return Apartment::setPrePretty($price) . ' ' . $currencyName . ' (' . Seasonalprices::rentalPeriodNames($price_type) . ')';
            }
        } else
            return '-';
    }

    public function search()
    {
        $criteria = new CDbCriteria;
        $criteria->compare('id', $this->id);
        $criteria->compare('apartment_id', $this->apartment_id);

        return new CustomActiveDataProvider($this, array(
            'criteria' => $criteria,
            'sort' => array('defaultOrder' => $this->getTableAlias() . '.sorter ASC'),
            'pagination' => array(
                'pageSize' => param('adminPaginationPageSize', 20),
            ),
        ));
    }

    public function setDefaults()
    {
        /*foreach (Lang::getActiveLangs() as $lang) {
            $this->{'name_' . $lang} = '-';
        }*/
    }

    public static function isUserAd($apartmentId = null, $ownerId = null)
    {
        if ($apartmentId && $ownerId) {
            if (Apartment::model()->findByAttributes(array('id' => $apartmentId, 'owner_id' => $ownerId))) {
                return true;
            }
        }
        return false;
    }

    public static function rentalPeriodNames($period = null)
    {
        $array = array();

        $array[Apartment::PRICE_PER_HOUR] = tt('hour', 'seasonalprices');
        $array[Apartment::PRICE_PER_DAY] = tt('day', 'seasonalprices');
        $array[Apartment::PRICE_PER_WEEK] = tt('week', 'seasonalprices');
        $array[Apartment::PRICE_PER_MONTH] = tt('month', 'seasonalprices');

        if (is_numeric($period)) {
            if (array_key_exists($period, $array))
                return $array[$period];
            elseif (array_key_exists($period, HApartment::getPriceArray(NULL, true)))
                return HApartment::getPriceName($period);
            else
                return '-';
        }

        return $array;
    }

    public static function getMinSorters($apartmentId)
    {
        $sql = 'SELECT MIN(sorter) FROM {{seasonal_prices}} WHERE apartment_id = ' . (int)$apartmentId;
        return (int)Yii::app()->db->createCommand($sql)->queryScalar();
    }

    public static function getMaxSorters($apartmentId)
    {
        $sql = 'SELECT MAX(sorter) FROM {{seasonal_prices}} WHERE apartment_id = ' . (int)$apartmentId;
        return (int)Yii::app()->db->createCommand($sql)->queryScalar();
    }
}
